// content_script.js — updated to support:
// 1) getProfile (existing)
// 2) capturePageText (NEW - required for popup.js AI)
// 3) safe clipboard write fallback

(function () {

  /********************************************************************
   * Helper: read text from selectors safely
   ********************************************************************/
  function qText(sel) {
    const el = document.querySelector(sel);
    return el ? el.innerText.trim() : "";
  }

  /********************************************************************
   * Normalized LinkedIn profile URL
   ********************************************************************/
  function canonicalProfileUrl() {
    try {
      const u = new URL(location.href);
      const clean = u.origin + u.pathname.replace(/\/+$/, "");
      return clean;
    } catch (e) {
      return location.href.split("?")[0].replace(/\/+$/, "");
    }
  }
  /********************************************************************
   * Linkedin sales navigator
   ********************************************************************/
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg.type === "getCanonicalProfileUrl") {
    let canonical = null;

    // 1️⃣ Primary: anchor pointing to /in/
    const linkEl = document.querySelector('a[href*="/in/"]');
    if (linkEl?.href) {
      canonical = linkEl.href;
    }

    // 2️⃣ Fallback: Open Graph meta
    if (!canonical) {
      const meta = document.querySelector('meta[property="og:url"]');
      if (meta?.content?.includes("/in/")) {
        canonical = meta.content;
      }
    }

    sendResponse({
      canonicalUrl: canonical
        ? canonical.split("?")[0].replace(/\/+$/, "")
        : null
    });

    return true; // keep channel open
  }
});


  /********************************************************************
   * MAIN SCRAPER (getProfile)
   * Uses multiple fallback selectors (LinkedIn changes often)
   ********************************************************************/
  function readProfileFields() {
    let fullName =
      qText("h1") ||
      qText(".pv-text-details__left-panel h1") ||
      qText(".text-heading-xlarge") ||
      "";

    let headline =
      qText(".pv-text-details__left-panel .text-body-medium") ||
      qText(".text-body-medium.break-words") ||
      qText(".text-body-medium") ||
      "";

    let location =
      qText(".pv-text-details__left-panel .text-body-small.inline") ||
      qText(".pv-text-details__left-panel .text-body-small") ||
      qText(".text-body-small") ||
      "";

    // Extract company & position from "Experience" top card
    let position = "";
    let company = "";

    const xpBlock =
      document.querySelector(".pv-top-card--experience-list") ||
      document.querySelector("section#experience-section") ||
      document.querySelector("section.pv-profile-section.experience-section");

    if (xpBlock) {
      const lines = xpBlock.innerText.split("\n").filter(Boolean);
      if (lines[0]) position = lines[0].trim();
      if (lines[1]) company = lines[1].trim();
    }

    // Headline fallback: "<role> at <company>"
    if (headline && !company) {
      const m = headline.match(/(.+)\s+at\s+(.+)/i);
      if (m) {
        position = m[1].trim();
        company = m[2].trim();
      }
    }

    const connectionStatus = (() => {
      for (const el of document.querySelectorAll("button, a")) {
        const t = (el.innerText || "").trim().toLowerCase();
        if (t === "message") return "connected";
        if (t === "connect") return "not_connected";
        if (t.includes("pending")) return "pending";
        if (t === "follow") return "not_connected";
      }
      return "unknown";
    })();

    const parts = (fullName || "").split(/\s+/).filter(Boolean);
    const firstName = parts[0] || "";
    const lastName = parts.slice(1).join(" ");

    return {
      profileUrl: canonicalProfileUrl(),
      fullName,
      firstName,
      lastName,
      headline,
      company,
      position,
      location,
      connectionStatus
    };
  }

  /********************************************************************
   * NEW — CAPTURE VISIBLE TEXT FOR AI
   ********************************************************************/
  function capturePageText() {
    try {
      const text = document.body.innerText || "";
      return text.trim();
    } catch (e) {
      return "";
    }
  }

  /********************************************************************
   * MESSAGE LISTENER (popup.js → content_script.js)
   ********************************************************************/
  chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    // 1) popup asks for profile
    if (msg.type === "getProfile") {
      sendResponse({ ok: true, profile: readProfileFields() });
      return true;
    }

    // 2) popup ping (checks if content script loaded)
    if (msg.type === "ping") {
      sendResponse({ ok: true });
      return true;
    }

    // 3) popup requests full page text
    if (msg.type === "capturePageText") {
      const text = capturePageText();
      sendResponse({ success: true, text });
      return true;
    }

    // 4) popup wants us to copy text safely
    if (msg.action === "copyText") {
      try {
        navigator.clipboard.writeText(msg.text).then(() => {
          sendResponse({ success: true });
        }).catch(() => {
          // fallback (textarea)
          const ta = document.createElement("textarea");
          ta.value = msg.text;
          ta.style.position = "fixed";
          ta.style.top = "-9999px";
          document.body.appendChild(ta);
          ta.select();
          document.execCommand("copy");
          ta.remove();
          sendResponse({ success: true });
        });
      } catch (e) {
        sendResponse({ success: false, error: e.message });
      }
      return true;
    }
  });

})();
