// icp_setup.js — Clean single version

// ── Storage helpers ──
function getLS(key) {
  return new Promise(r => chrome.storage.local.get(key, o => r(o[key])));
}
function setLS(obj) {
  return new Promise(r => chrome.storage.local.set(obj, () => r()));
}

// ── Toast ──
function showToast(msg) {
  const t = document.getElementById('toast');
  if (!t) return;
  t.textContent = msg;
  t.classList.add('show');
  setTimeout(() => t.classList.remove('show'), 2200);
}

// ── Industry list ──
const LINKEDIN_INDUSTRIES = [
  "Accounting","Airlines/Aviation","Alternative Dispute Resolution","Alternative Medicine",
  "Animation","Apparel & Fashion","Architecture & Planning","Arts & Crafts","Automotive",
  "Aviation & Aerospace","Banking","Biotechnology","Broadcast Media","Building Materials",
  "Business Supplies & Equipment","Capital Markets","Chemicals","Civic & Social Organization",
  "Civil Engineering","Commercial Real Estate","Computer & Network Security","Computer Games",
  "Computer Hardware","Computer Networking","Computer Software","Construction",
  "Consumer Electronics","Consumer Goods","Consumer Services","Cosmetics","Dairy",
  "Defense & Space","Design","Education Management","E-Learning",
  "Electrical/Electronic Manufacturing","Entertainment","Environmental Services",
  "Events Services","Executive Office","Facilities Services","Farming","Financial Services",
  "Fine Art","Fishery","Food & Beverages","Food Production","Fund-Raising","Furniture",
  "Gambling & Casinos","Glass/Ceramics/Concrete","Government Administration",
  "Government Relations","Graphic Design","Health/Wellness & Fitness","Higher Education",
  "Hospital/Health Care","Hospitality","Human Resources","Import/Export",
  "Individual & Family Services","Industrial Automation","Information Services",
  "Information Technology & Services","Insurance","International Affairs",
  "International Trade & Development","Internet","Investment Banking","Investment Management",
  "Judiciary","Law Enforcement","Law Practice","Legal Services","Legislative Office",
  "Leisure/Travel & Tourism","Libraries","Logistics & Supply Chain","Luxury Goods & Jewelry",
  "Machinery","Management Consulting","Maritime","Marketing & Advertising","Market Research",
  "Mechanical/Industrial Engineering","Media Production","Medical Devices","Medical Practice",
  "Mental Health Care","Military","Mining & Metals","Motion Pictures & Film",
  "Museums & Institutions","Music","Nanotechnology","Newspapers",
  "Non-Profit Organization Management","Oil & Energy","Online Publishing",
  "Outsourcing/Offshoring","Package/Freight Delivery","Packaging & Containers",
  "Paper & Forest Products","Performing Arts","Pharmaceuticals","Philanthropy","Photography",
  "Plastics","Political Organization","Primary/Secondary Education","Printing",
  "Professional Training & Coaching","Program Development","Public Policy",
  "Public Relations & Communications","Public Safety","Publishing","Railroad Manufacture",
  "Ranching","Real Estate","Recreational Facilities & Services","Religious Institutions",
  "Renewables & Environment","Research","Restaurants","Retail","Security & Investigations",
  "Semiconductors","Shipbuilding","Sporting Goods","Sports","Staffing & Recruiting",
  "Supermarkets","Telecommunications","Textiles","Think Tanks","Tobacco",
  "Translation & Localization","Transportation/Trucking/Railroad","Utilities",
  "Venture Capital & Private Equity","Veterinary","Warehousing","Wholesale",
  "Wine & Spirits","Wireless","Writing & Editing"
];

const REGIONS = [
  "Global","US - East Coast","US - West Coast","US - Midwest","US - South",
  "Canada","UK","Europe","DACH (Germany/Austria/Switzerland)","Nordics","Baltics",
  "Benelux","Eastern Europe","Middle East","Africa","India","South Asia",
  "Southeast Asia","APAC","Australia/New Zealand","LATAM","Brazil","Mexico"
];

// ══════════════════════════════════════════════════
// Multi-select component
// Closes on: outside click, Tab key, Escape key
// ══════════════════════════════════════════════════
function setupMultiSelect(containerEl, list) {
  let selected = [];
  let dropdownEl = null;

  function renderTags() {
    containerEl.innerHTML = selected.length
      ? selected.map(s =>
          `<span class="tag">${s} <span class="tag-remove" data-val="${s}">×</span></span>`
        ).join('')
      : '<span class="multi-box-placeholder">Click to select...</span>';

    // Tag remove buttons
    containerEl.querySelectorAll('.tag-remove').forEach(btn => {
      btn.addEventListener('click', e => {
        e.stopPropagation();
        selected = selected.filter(v => v !== btn.dataset.val);
        renderTags();
      });
    });
  }

  function closeDropdown() {
    if (dropdownEl) {
      dropdownEl.remove();
      dropdownEl = null;
      containerEl.classList.remove('active');
    }
  }

  function openDropdown() {
    if (dropdownEl) { closeDropdown(); return; }

    containerEl.classList.add('active');

    dropdownEl = document.createElement('div');
    dropdownEl.style.cssText = `
      position: fixed;
      background: #fff;
      border: 1.5px solid #0077b5;
      border-radius: 8px;
      padding: 8px;
      max-height: 240px;
      overflow-y: auto;
      z-index: 999999;
      box-shadow: 0 8px 24px rgba(0,0,0,0.12);
      min-width: 280px;
    `;

    // Position under container
    const rect = containerEl.getBoundingClientRect();
    dropdownEl.style.left = rect.left + 'px';
    dropdownEl.style.top  = (rect.bottom + 4) + 'px';
    dropdownEl.style.width = rect.width + 'px';

    // Search input
    const search = document.createElement('input');
    search.placeholder = 'Search...';
    search.style.cssText = 'width:100%; margin-bottom:6px; padding:7px 10px; border:1.5px solid #e2e8f0; border-radius:6px; font-family:inherit; font-size:13px; box-sizing:border-box;';
    dropdownEl.appendChild(search);

    function renderOptions(filter = '') {
      dropdownEl.querySelectorAll('.dd-opt').forEach(e => e.remove());
      const filtered = list.filter(i => i.toLowerCase().includes(filter.toLowerCase()));
      filtered.forEach(item => {
        const opt = document.createElement('div');
        opt.className = 'dd-opt';
        opt.style.cssText = `
          padding: 7px 10px;
          cursor: pointer;
          border-radius: 5px;
          font-size: 13px;
          color: #111318;
          display: flex;
          align-items: center;
          gap: 8px;
        `;
        const isSelected = selected.includes(item);
        if (isSelected) {
          opt.style.background = '#e8f3f9';
          opt.style.color = '#0077b5';
          opt.style.fontWeight = '600';
        }
        opt.innerHTML = `<span style="font-size:11px;">${isSelected ? '✓' : '○'}</span> ${item}`;
        opt.addEventListener('mousedown', e => {
          e.preventDefault(); // prevent blur
          if (selected.includes(item)) {
            selected = selected.filter(v => v !== item);
          } else {
            selected.push(item);
          }
          renderTags();
          renderOptions(search.value);
        });
        dropdownEl.appendChild(opt);
      });
    }

    search.addEventListener('input', () => renderOptions(search.value));

    // Close on Tab or Escape
    search.addEventListener('keydown', e => {
      if (e.key === 'Tab' || e.key === 'Escape') {
        e.preventDefault();
        closeDropdown();
      }
    });

    renderOptions();
    document.body.appendChild(dropdownEl);

    // Focus search
    setTimeout(() => search.focus(), 10);

    // Close on outside click
    function onOutsideClick(e) {
      if (!dropdownEl) return;
      if (!dropdownEl.contains(e.target) && !containerEl.contains(e.target)) {
        closeDropdown();
        document.removeEventListener('mousedown', onOutsideClick);
      }
    }
    document.addEventListener('mousedown', onOutsideClick);
  }

  containerEl.addEventListener('click', openDropdown);

  // Keyboard accessibility
  containerEl.addEventListener('keydown', e => {
    if (e.key === 'Enter' || e.key === ' ') { e.preventDefault(); openDropdown(); }
    if (e.key === 'Escape') closeDropdown();
  });

  renderTags();

  return {
    getSelected: () => [...selected],
    setSelected: (arr) => { selected = Array.isArray(arr) ? [...arr] : []; renderTags(); }
  };
}

// ══════════════════════════════════════════════════
// MAIN
// ══════════════════════════════════════════════════
document.addEventListener('DOMContentLoaded', async () => {

  // Init multi-selects
  const industrySelect = setupMultiSelect(
    document.getElementById('industrySelect'),
    LINKEDIN_INDUSTRIES
  );
  const geoSelect = setupMultiSelect(
    document.getElementById('geoSelect'),
    REGIONS
  );

  // Load saved settings
  const settings = (await getLS('icpSettings')) || {};

  if (settings.industries)  industrySelect.setSelected(settings.industries);
  if (settings.regions)     geoSelect.setSelected(settings.regions);

  // Restore persona checkboxes + visual state
  document.querySelectorAll('.persona').forEach(cb => {
    const checked = settings.personas?.includes(cb.value) || false;
    cb.checked = checked;
    const item = cb.closest('.checkbox-item');
    if (item) item.classList.toggle('checked', checked);
  });

  // Restore other fields
  const g = id => document.getElementById(id);

  if (g('persona_custom'))    g('persona_custom').value    = settings.customPersona || '';
  if (g('geo_custom'))        g('geo_custom').value        = settings.customRegion  || '';
  if (g('offer_service'))     g('offer_service').value     = settings.offer?.service     || '';
  if (g('offer_pain'))        g('offer_pain').value        = settings.offer?.pain        || '';
  if (g('offer_tone'))        g('offer_tone').value        = settings.offer?.tone        || 'Professional';
  if (g('offer_cta'))         g('offer_cta').value         = settings.offer?.cta         || 'Book a quick call';
  if (g('offer_exclusions'))  g('offer_exclusions').value  = settings.offer?.exclusions  || '';

  // Restore company size + stage multi-selects
  if (settings.companySize) {
    Array.from(g('icp_size').options).forEach(o => {
      o.selected = settings.companySize.includes(o.value);
    });
  }
  if (settings.companyStage) {
    Array.from(g('icp_stage').options).forEach(o => {
      o.selected = settings.companyStage.includes(o.value);
    });
  }

  // Restore AI goal radio
  if (settings.goal) {
    const goalRadio = document.querySelector(`input[name="ai_goal"][value="${settings.goal}"]`);
    if (goalRadio) {
      goalRadio.checked = true;
      goalRadio.closest('.provider-card')?.classList.add('selected');
    }
  }

  // ── SAVE ──
  g('saveIcpBtn').addEventListener('click', async () => {
    const personas = Array.from(document.querySelectorAll('.persona'))
      .filter(cb => cb.checked)
      .map(cb => cb.value);

    const goal = document.querySelector('input[name="ai_goal"]:checked')?.value || 'pipeline';

    const icpData = {
      industries:   industrySelect.getSelected(),
      regions:      geoSelect.getSelected(),
      customRegion: (g('geo_custom')?.value || '').trim(),
      personas,
      customPersona: (g('persona_custom')?.value || '').trim(),
      companySize:  Array.from(g('icp_size').selectedOptions).map(o => o.value),
      companyStage: Array.from(g('icp_stage').selectedOptions).map(o => o.value),
      offer: {
        service:    (g('offer_service')?.value    || '').trim(),
        pain:       (g('offer_pain')?.value       || '').trim(),
        tone:       g('offer_tone')?.value        || 'Professional',
        cta:        g('offer_cta')?.value         || 'Book a quick call',
        exclusions: (g('offer_exclusions')?.value || '').trim()
      },
      goal,
      updatedAt: new Date().toISOString()
    };

    await setLS({ icpSettings: icpData });

    // Visual feedback
    const btn = g('saveIcpBtn');
    const status = g('saveStatus');
    btn.textContent = '✓ Saved!';
    if (status) { status.textContent = '✓ Settings saved'; status.classList.add('saved'); }
    showToast('✓ ICP settings saved');
    setTimeout(() => {
      btn.textContent = 'Save Settings';
      if (status) { status.textContent = 'All changes saved'; }
    }, 2000);
  });

  // Persona checkbox visual toggle
  document.querySelectorAll('.checkbox-item').forEach(item => {
    const cb = item.querySelector('input[type="checkbox"]');
    if (!cb) return;
    item.addEventListener('click', () => {
      setTimeout(() => item.classList.toggle('checked', cb.checked), 0);
    });
  });

  // Provider card visual toggle
  document.querySelectorAll('.provider-card').forEach(card => {
    const radio = card.querySelector('input[type="radio"]');
    if (!radio) return;
    radio.addEventListener('change', () => {
      const name = radio.name;
      document.querySelectorAll(`input[name="${name}"]`).forEach(r => {
        r.closest('.provider-card')?.classList.remove('selected');
      });
      if (radio.checked) card.classList.add('selected');
    });
  });

});