/******************************************************************************
 * POPUP.JS — NO AI VERSION (Sequences + Templates + Scheduling only)
 *****************************************************************************/

console.log("%c[popup.js] Loaded — AI removed, sequencing only", "color:#0a0;font-weight:bold;");
// ─── AUTH CHECK ───────────────────────────────────────────────────────────────

async function initAuth() {
  const loginBlock  = document.getElementById('loginBlock');
  const mainBlock   = document.getElementById('mainBlock');
  const authBar     = document.getElementById('authBar');
  const userEmailEl = document.getElementById('userEmail');
  const creditsBadge = document.getElementById('creditsNum');
  const sendLoginBtn = document.getElementById('sendLoginBtn');
  const loginEmail   = document.getElementById('loginEmail');
  const loginSentMsg = document.getElementById('loginSentMsg');
  const logoutBtn    = document.getElementById('logoutBtn');

  // Check if session exists and is valid
  const session = await syncClient.getSession();

  if (!session) {
    // Not logged in — show login screen
    loginBlock.style.display = 'block';
    mainBlock.style.display  = 'none';
    authBar.style.display    = 'none';

    // Send magic link
    sendLoginBtn.addEventListener('click', async () => {
      const email = loginEmail.value.trim();
      if (!email || !email.includes('@')) {
        alert('Please enter a valid email address');
        return;
      }

      sendLoginBtn.disabled = true;
      sendLoginBtn.textContent = 'Sending...';

      try {
        const res = await fetch('https://api.linkedvue.com/auth/magic-link', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ email })
        });

        const data = await res.json();

        if (data.success) {
          loginSentMsg.style.display = 'block';
          sendLoginBtn.textContent = 'Link Sent ✓';
        } else {
          alert('Error: ' + (data.error || 'Could not send link'));
          sendLoginBtn.disabled = false;
          sendLoginBtn.textContent = 'Send Login Link';
        }
      } catch (err) {
        alert('Network error. Please try again.');
        sendLoginBtn.disabled = false;
        sendLoginBtn.textContent = 'Send Login Link';
      }
    });

    return false; // stop popup.js from running main logic
  }

  // Logged in — validate session with /me
  try {
    const res = await fetch('https://api.linkedvue.com/me', {
      headers: { 'Authorization': `Bearer ${session}` }
    });

    if (!res.ok) {
      // Session expired — clear and show login
      await syncClient.clearSession();
      loginBlock.style.display = 'block';
      mainBlock.style.display  = 'none';
      authBar.style.display    = 'none';
      return false;
    }

    const user = await res.json();

    // Show main UI
    loginBlock.style.display = 'none';
    mainBlock.style.display  = 'block';
    authBar.style.display    = 'flex';

    // Populate auth bar
    userEmailEl.textContent  = user.email;
    creditsBadge.textContent = `${user.credits_remaining} credits`;

    // Logout button
    logoutBtn.addEventListener('click', async () => {
      await fetch('https://api.linkedvue.com/auth/session', {
        method: 'DELETE',
        headers: { 'Authorization': `Bearer ${session}` }
      });
      await syncClient.clearSession();
      loginBlock.style.display = 'block';
      mainBlock.style.display  = 'none';
      authBar.style.display    = 'none';
    });

    // Pull stale data on open
    syncClient.pullIfStale();

    return true; // continue with main popup logic

  } catch (err) {
    // Network error — show main UI anyway (offline mode)
    loginBlock.style.display = 'none';
    mainBlock.style.display  = 'block';
    authBar.style.display    = 'none';
    return true;
  }
}
document.addEventListener("DOMContentLoaded", async () => {
const authed = await initAuth();
  if (!authed) return;

  // Add this near the top of popup.js after DOMContentLoaded
chrome.storage.onChanged.addListener((changes, area) => {
  if (area === 'local' && changes['linkedvue_session_id']?.newValue) {
    // Session just got saved — reinit auth
    initAuth();
  }

  // Near top of popup.js DOMContentLoaded handler
chrome.storage.onChanged.addListener((changes, area) => {
  if (area === 'local' && changes['linkedvue_session_id']?.newValue) {
    initAuth();
  }
});
});
/**************************************************************************
 * LOCAL STORAGE HELPERS
 **************************************************************************/
function getLS(key) {
  return new Promise(resolve => {
    chrome.storage.local.get(key, result => {
      resolve(key ? result[key] : result);
    });
  });
}

function setLS(obj) {
  return new Promise(resolve => {
    chrome.storage.local.set(obj, () => resolve());
  });
}
// ===============================
// AI UI ELEMENTS
// ===============================
const rawProfileTextEl = document.getElementById("rawProfileText");
const generateAiBtn = document.getElementById("generateAiBtn");
const icpScoreValueEl = document.getElementById("icpScoreValue");
const aiMessageBoxEl = document.getElementById("aiMessageBox");
// ===============================
// SAFE JSON PARSER
// ===============================
function safeParseAiJson(raw) {
  try {
    return JSON.parse(raw);
  } catch {
    const match = raw.match(/\{[\s\S]*\}/);
    if (!match) return null;
    try {
      return JSON.parse(match[0]);
    } catch {
      return null;
    }
  }
}

// ===============================
// AI PROMPT BUILDER (STEP 2)
// ===============================
function buildAiPrompt({ icp, rawProfileText }) {
  return `
You are a GTM assistant helping generate outbound LinkedIn messages.

RULES (MANDATORY):
- Respond in VALID JSON ONLY.
- Do NOT include explanations.
- Do NOT include markdown.
- Do NOT include extra keys.
- The message must be 80 to 100 words.
- Message tone: practical, founder-to-founder, non-salesy.

TASKS:
1. Extract key facts from the LinkedIn profile text.
2. Generate ONE clear LinkedIn message aligned with the ICP goal.

ICP SETTINGS:
Industries: ${icp.industries?.join(", ") || "—"}
Regions: ${icp.regions?.join(", ") || "—"}
Personas: ${icp.personas?.join(", ") || "—"}
Goal: ${icp.goal || "—"}
Offer:
- Service: ${icp.offer?.service || ""}
- Pain: ${icp.offer?.pain || ""}
- CTA: ${icp.offer?.cta || ""}

LINKEDIN PROFILE TEXT:
"""
${rawProfileText}
"""

RESPONSE FORMAT (JSON ONLY):
{
  "extracted": {
    "persona": "",
    "industry": "",
    "region": "",
    "companyType": ""
  },
  "message": ""
}
`.trim();
}

/**************************************************************************
 * UTILITIES
 **************************************************************************/
function sanitizeForStorage(text) {
  if (typeof text !== "string") return "";
  return text.replace(/[\u0000-\u001F\u007F]/g, "").trim();
}

function showToast(msg) {
  const t = document.getElementById("toast");
  if (!t) return;
  t.innerText = msg;
  t.style.opacity = "1";
  setTimeout(() => { t.style.opacity = "0"; }, 1800);
}

function getNextDaySameTime() {
  const d = new Date();
  d.setDate(d.getDate() + 1);
  d.setSeconds(0, 0);
  return d;
}

function toLocalDatetimeValue(date) {
  const pad = n => String(n).padStart(2, "0");
  return `${date.getFullYear()}-${pad(date.getMonth()+1)}-${pad(date.getDate())}T${pad(date.getHours())}:${pad(date.getMinutes())}`;
}

/**************************************************************************
 * UI REFERENCES
 **************************************************************************/
const notLinkedIn = document.getElementById("notLinkedIn");
const profileBlock = document.getElementById("profileBlock");

const p_name = document.getElementById("p_name");
const p_headline = document.getElementById("p_headline");
const p_company_loc = document.getElementById("p_company_loc");
const p_status = document.getElementById("p_status");
const p_url = document.getElementById("p_url");

const p_firstName = document.getElementById("p_firstName");
const p_email = document.getElementById("p_email");
const p_notes = document.getElementById("p_notes");

const saveDetailsBtn = document.getElementById("saveEmailNotesBtn");

const scheduleBtn = document.getElementById("scheduleBtn");
const schedulePanel = document.getElementById("schedulePanel");
const scheduleInput = document.getElementById("scheduleInput");
const scheduleCalendar = document.getElementById("scheduleCalendar");
const confirmScheduleBtn = document.getElementById("confirmScheduleBtn");

const templateSelect = document.getElementById("templateSelect");
const applyTplBtn = document.getElementById("applyTplBtn");
const copyNextBtn = document.getElementById("copyNextBtn");
const markSentBtn = document.getElementById("markSentBtn");
const editSeqBtn = document.getElementById("editSeqBtn");
const deleteSeqBtn = document.getElementById("deleteSeqBtn");
const openDashboardBtn = document.getElementById("openDashboardBtn");

const seqNameEl = document.getElementById("seqName");
const nextActionEl = document.getElementById("nextAction");

/**************************************************************************
 * STEP 1 — ENSURE LINKEDIN PROFILE TAB
 **************************************************************************/
const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

if (!tab || !tab.url || !tab.url.includes("linkedin.com")) {
  notLinkedIn.style.display = "block";
  profileBlock.style.display = "none";
  return;
}

/**************************************************************************
 * STEP 2 — REQUEST PROFILE FROM CONTENT SCRIPT
 **************************************************************************/
async function requestProfile() {
  try {
    const res = await chrome.tabs.sendMessage(tab.id, { type: "getProfile" });
    if (res && res.ok) return res.profile;
  } catch {}

  try {
    await chrome.scripting.executeScript({
      target: { tabId: tab.id },
      files: ["content_script.js"]
    });
    const retry = await chrome.tabs.sendMessage(tab.id, { type: "getProfile" });
    if (retry && retry.ok) return retry.profile;
  } catch (e) {
    console.error("Profile scrape failed:", e);
  }
  return null;
}

const profile = await requestProfile();
if (!profile) {
  notLinkedIn.style.display = "block";
  profileBlock.style.display = "none";
  return;
}

/**************************************************************************
 * POPULATE PROFILE UI
 **************************************************************************/
profileBlock.style.display = "block";
notLinkedIn.style.display = "none";

p_name.innerText = profile.fullName || "(no name)";
p_headline.innerText = profile.headline || "";
p_company_loc.innerText =
  (profile.company ? profile.company + " • " : "") + (profile.location || "");
p_status.innerText = profile.connectionStatus || "unknown";
p_url.href = profile.profileUrl;
p_url.innerText = profile.profileUrl;

// Avatar initial
const avatarEl = document.getElementById("profileAvatar");
if (avatarEl) {
  avatarEl.innerText = (profile.firstName || profile.fullName || "?")[0].toUpperCase();
}

// Status badge
const statusBadgeEl = document.getElementById("p_status_badge");
if (statusBadgeEl) {
  const s = profile.connectionStatus || "";
  statusBadgeEl.innerText = s || "unknown";
  if (s.toLowerCase().includes("connect")) statusBadgeEl.classList.add("connected");
}

// --------------------------------------------------
// Canonical LinkedIn profile URL (HARDENED & STABLE)
// --------------------------------------------------

async function getStableLinkedInProfileUrl(tabId) {
  // LinkedIn often redirects via tscp-serving before landing on /in/
  // We retry a few times to let the URL stabilize
  for (let i = 0; i < 5; i++) {
    try {
      const t = await chrome.tabs.get(tabId);
      const url = t?.url;

      if (
        url &&
        url.startsWith("https://www.linkedin.com/") &&
        url.includes("/in/") &&
        !url.includes("tscp-serving")
      ) {
        return url.split("?")[0].replace(/\/+$/, "");
      }
    } catch (e) {
      // ignore and retry
    }

    // wait before retrying
    await new Promise(r => setTimeout(r, 300));
  }

  return null;
}

const rawUrl = await getStableLinkedInProfileUrl(tab.id);

if (!rawUrl) {
  notLinkedIn.style.display = "block";
  profileBlock.style.display = "none";
  return;
}

let urlKey = null;

try {
  const u = new URL(rawUrl);

  // STRICT: only real LinkedIn profile URLs
  if (
    u.hostname === "www.linkedin.com" &&
    u.pathname.startsWith("/in/")
  ) {
    urlKey = `${u.origin}${u.pathname}`;
  }
} catch (e) {
  // invalid URL
}

if (!urlKey) {
  notLinkedIn.style.display = "block";
  profileBlock.style.display = "none";
  return;
}

// 🔒 ALWAYS overwrite anything scraped or injected
profile.profileUrl = urlKey;
p_url.href = urlKey;
p_url.innerText = urlKey;


/**************************************************************************
 * LOAD TEMPLATES + EXISTING PROFILE RECORD
 **************************************************************************/
const templates = (await getLS("templates")) || {};
const profiles = (await getLS("profiles")) || {};

templateSelect.innerHTML = '<option value="">— choose template —</option>';
Object.keys(templates).forEach(name => {
  const op = document.createElement("option");
  op.value = name;
  op.innerText = name;
  templateSelect.appendChild(op);
});

const rec = profiles[urlKey] || null;

if (!rec) {
  seqNameEl.innerText = "—";
  nextActionEl.innerText = "No sequence saved";
  p_firstName.value = profile.firstName || "";
  p_email.value = "";
  p_notes.value = "";
} else {
  seqNameEl.innerText = rec.templateUsed || "—";
  p_firstName.value = rec.firstName || profile.firstName || "";
  p_email.value = rec.email || "";
  p_notes.value = rec.notes || "";
  templateSelect.value = rec.templateUsed || "";

  const next =
    !rec.sentDates?.msg1 ? "Send Message 1" :
    !rec.sentDates?.msg2 ? "Send Message 2" :
    !rec.sentDates?.msg3 ? "Send Message 3" :
    "Sequence completed";

nextActionEl.innerText = next;
}
updateSequenceDots(rec);
// --------------------------------------------------
// Restore AI data on popup open (STEP 4.6)
// --------------------------------------------------
if (rec?.aiMessage && aiMessageBoxEl) {
  aiMessageBoxEl.value = rec.aiMessage;
}

if (rec?.aiExtractedFacts) {
  window.__aiExtractedFacts = rec.aiExtractedFacts;
}

if (rec?.icpScore && icpScoreValueEl) {
  icpScoreValueEl.textContent = rec.icpScore;
}


/**************************************************************************
 * MERGE TAG RESOLVER
 **************************************************************************/
function resolveMerge(text, p) {
  if (!text) return "";
  return text
    .replace(/{{\s*first_name\s*}}/gi, p.firstName || "")
    .replace(/{{\s*last_name\s*}}/gi, p.lastName || "")
    .replace(/{{\s*full_name\s*}}/gi, p.fullName || "")
    .replace(/{{\s*headline\s*}}/gi, p.headline || "")
    .replace(/{{\s*company\s*}}/gi, p.company || "")
    .replace(/{{\s*position\s*}}/gi, p.position || "")
    .replace(/{{\s*location\s*}}/gi, p.location || "")
    .replace(/{{\s*today\s*}}/gi, new Date().toLocaleDateString());
}

/**************************************************************************
 * SAVE DETAILS
 **************************************************************************/
saveDetailsBtn?.addEventListener("click", async () => {
  try {

    // 🔒 URL SAFETY GUARD
    if (!urlKey || !urlKey.includes("/in/")) {
      console.warn("Blocked save (details) for invalid URL:", urlKey);
      return;
    }

    const profilesAll = (await getLS("profiles")) || {};

    // 🔒 IMPORTANT: merge with existing record, never replace
    let r = profilesAll[urlKey] || {
      profileUrl: urlKey,
      dateCaptured: new Date().toISOString(),
      sentDates: { msg1: null, msg2: null, msg3: null }
    };

    r.firstName = p_firstName.value.trim() || profile.firstName || "";
    r.email = p_email.value.trim() || "";
    r.notes = sanitizeForStorage(p_notes.value || "");
    r.lastUpdated = new Date().toISOString();

    // ✅ MERGE BACK (preserves icpScore, aiMessage, etc.)
    profilesAll[urlKey] = {
      ...profilesAll[urlKey],
      ...r
    };

    await setLS({ profiles: profilesAll });
    showToast("Details saved");
syncClient.pushEvent('profile_saved');
  } catch (e) {
    alert("Save failed: " + e.message);
  }
});

/**************************************************************************
 * APPLY TEMPLATE
 **************************************************************************/
applyTplBtn?.addEventListener("click", async () => {

  // 🔒 URL SAFETY GUARD
  if (!urlKey || !urlKey.includes("/in/")) {
    console.warn("Blocked save (template) for invalid URL:", urlKey);
    return;
  }

  const selected = templateSelect.value;
  if (!selected) return showToast("Choose a template");

  const tpl = templates[selected];
  if (!tpl) return showToast("Template not found");

  const profilesAll = (await getLS("profiles")) || {};

  // 🔒 MERGE with existing profile instead of overwriting
  let r = profilesAll[urlKey] || {
    profileUrl: urlKey,
    dateCaptured: new Date().toISOString(),
    sentDates: { msg1: null, msg2: null, msg3: null }
  };

  // update profile basics (do not nuke existing data)
  r.profileUrl = urlKey;
  r.lastUpdated = new Date().toISOString();
  r.fullName = profile.fullName || r.fullName || "";
  r.firstName = p_firstName.value || r.firstName || "";
  r.lastName = profile.lastName || r.lastName || "";
  r.headline = profile.headline || r.headline || "";
  r.company = profile.company || r.company || "";
  r.position = profile.position || r.position || "";
  r.location = profile.location || r.location || "";
  r.email = p_email.value || r.email || "";
  r.notes = sanitizeForStorage(p_notes.value || r.notes || "");

  // apply template messages
  r.templateUsed = selected;
  r.messages = JSON.parse(JSON.stringify(tpl.messages || {}));

  // ✅ NEW: copy single delay into profile
  r.sequenceDelayDays = tpl.delayDays ?? 3;

// reset sequence state ONLY on explicit apply
r.sentDates = { msg1: null, msg2: null, msg3: null };
r.status = "ready";

// ✅ CRITICAL: initialize queue date for Message 1
r.nextDueDate = new Date().toISOString();


  profilesAll[urlKey] = {
    ...profilesAll[urlKey],
    ...r
  };

  await setLS({ profiles: profilesAll });

  seqNameEl.innerText = selected;
  nextActionEl.innerText = "Send Message 1";
  showToast("Template applied");
});

/**************************************************************************
 * COPY NEXT MESSAGE
 **************************************************************************/
copyNextBtn?.addEventListener("click", async () => {
  const profilesAll = (await getLS("profiles")) || {};
  const r = profilesAll[urlKey];
  if (!r) return showToast("No sequence");

  const nextKey =
    !r.sentDates?.msg1 ? "msg1" :
    !r.sentDates?.msg2 ? "msg2" :
    !r.sentDates?.msg3 ? "msg3" : null;

  if (!nextKey) return showToast("Sequence complete");

  const msg = resolveMerge(r.messages?.[nextKey] || "", r);
  await navigator.clipboard.writeText(msg);
  showToast("Message copied");
});

/**************************************************************************
 * MARK SENT
 **************************************************************************/
markSentBtn?.addEventListener("click", async () => {

  // 🔒 URL SAFETY GUARD
  if (!urlKey || !urlKey.includes("/in/")) {
    console.warn("Blocked save (mark sent) for invalid URL:", urlKey);
    return;
  }

  const profilesAll = (await getLS("profiles")) || {};

  const r = profilesAll[urlKey];
  if (!r) return;

const now = new Date();
const nowISO = now.toISOString();

// determine delay (template-level or default)
const delayDays = Number(r.sequenceDelayDays ?? 3);
const nextDue = new Date(now.getTime() + delayDays * 86400000).toISOString();

if (!r.sentDates.msg1) {
  r.sentDates.msg1 = nowISO;
  r.nextDueDate = nextDue;
  r.status = "msg1_sent";
}
else if (!r.sentDates.msg2) {
  r.sentDates.msg2 = nowISO;
  r.nextDueDate = nextDue;
  r.status = "msg2_sent";
}
else if (!r.sentDates.msg3) {
  r.sentDates.msg3 = nowISO;
  r.nextDueDate = null; // sequence complete
  r.status = "completed";
}

r.lastUpdated = nowISO;
profilesAll[urlKey] = r;
await setLS({ profiles: profilesAll });

showToast("Marked as sent");
syncClient.pushEvent('message_sent');
updateSequenceDots(profilesAll[urlKey]);
});

/**************************************************************************
 * SCHEDULING
 **************************************************************************/

scheduleBtn?.addEventListener("click", async () => {
  schedulePanel.classList.toggle("open");
});

scheduleCalendar?.addEventListener("change", () => {
  scheduleInput.value = new Date(scheduleCalendar.value).toLocaleString();
});

confirmScheduleBtn?.addEventListener("click", async () => {

  // 🔒 URL SAFETY GUARD
  if (!urlKey || !urlKey.includes("/in/")) {
    console.warn("Blocked save (schedule) for invalid URL:", urlKey);
    return;
  }

  const when = new Date(scheduleCalendar.value);
  if (isNaN(when)) {
    alert("Invalid date");
    return;
  }

  const profilesAll = (await getLS("profiles")) || {};
  let r = profilesAll[urlKey] || { profileUrl: urlKey };

  // Encode URL for alarm-safe name
  const encodedUrl = btoa(urlKey);
  const alarmName = `schedule_${encodedUrl}`;

  // Persist scheduling metadata
  r.scheduledAt = when.getTime();
r.nextDueDate = new Date(when.getTime()).toISOString(); // ✅ dashboard uses this
r.scheduleAlarm = alarmName;
r.lastUpdated = new Date().toISOString();


  profilesAll[urlKey] = r;

  await setLS({
    profiles: profilesAll,
    last_schedule_datetime: when.getTime()
  });

  // Clear any existing alarm for this profile (idempotent)
  chrome.alarms.clear(alarmName, () => {
    chrome.alarms.create(alarmName, { when: when.getTime() });
  });

  showToast("Scheduled");
});


/**************************************************************************
 * DELETE SEQUENCE
 **************************************************************************/
deleteSeqBtn?.addEventListener("click", async () => {
  if (!confirm("Delete saved sequence?")) return;
  const profilesAll = (await getLS("profiles")) || {};
  delete profilesAll[urlKey];
  await setLS({ profiles: profilesAll });
  seqNameEl.innerText = "—";
  nextActionEl.innerText = "No sequence saved";
  showToast("Deleted");
});

/**************************************************************************
 * SEQUENCE PROGRESS DOTS
 **************************************************************************/
function updateSequenceDots(rec) {
  const dot1 = document.getElementById("dot1");
  const dot2 = document.getElementById("dot2");
  const dot3 = document.getElementById("dot3");
  const lbl1 = document.getElementById("lbl1");
  const lbl2 = document.getElementById("lbl2");
  const lbl3 = document.getElementById("lbl3");
  const conn1 = document.getElementById("conn1");
  const conn2 = document.getElementById("conn2");
  const badge = document.getElementById("seqBadge");

  // Reset all
  [dot1,dot2,dot3].forEach(d => d?.classList.remove("sent","next"));
  [lbl1,lbl2,lbl3].forEach(l => l?.classList.remove("active"));
  [conn1,conn2].forEach(c => c?.classList.remove("done"));

  if (!rec?.templateUsed) {
    if (badge) { badge.innerText = "No template"; badge.className = "section-badge"; }
    return;
  }

  const s = rec.sentDates || {};
  const sent = [s.msg1, s.msg2, s.msg3].filter(Boolean).length;

  if (s.msg1) { dot1?.classList.add("sent"); dot1.innerText = "✓"; conn1?.classList.add("done"); }
  else { dot1?.classList.add("next"); lbl1?.classList.add("active"); }

  if (s.msg2) { dot2?.classList.add("sent"); dot2.innerText = "✓"; conn2?.classList.add("done"); }
  else if (s.msg1) { dot2?.classList.add("next"); lbl2?.classList.add("active"); }

  if (s.msg3) { dot3?.classList.add("sent"); dot3.innerText = "✓"; }
  else if (s.msg2) { dot3?.classList.add("next"); lbl3?.classList.add("active"); }

  if (badge) {
    if (sent === 3) {
      badge.innerText = "Complete ✓";
      badge.className = "section-badge done";
    } else {
      badge.innerText = `${rec.templateUsed} · ${sent}/3`;
      badge.className = "section-badge active";
    }
  }
}

/**************************************************************************
 * OPEN DASHBOARD
 **************************************************************************/
openDashboardBtn?.addEventListener("click", () => {
  chrome.runtime.openOptionsPage();
});

/**************************************************************************
 * AI SECTION — Full replacement block for popup.js
 * Replaces the entire generateAiBtn click handler
 * Server handles: Groq API call + ICP scoring
 * Extension handles: display + local persistence only
 **************************************************************************/

generateAiBtn?.addEventListener("click", async () => {
  const rawText = rawProfileTextEl?.value?.trim();

  // ── Guard: need profile text ──
  if (!rawText || rawText.length < 50) {
    showToast("Paste more profile text first");
    return;
  }

  // ── Guard: need ICP settings ──
  const icpSettings = await getLS("icpSettings");
  if (!icpSettings) {
    showToast("Configure ICP settings first");
    return;
  }

  // ── Guard: need session ──
  const session = await syncClient.getSession?.();
  if (!session) {
    showToast("Please log in first");
    return;
  }

  // ── Loading state ──
  generateAiBtn.disabled = true;
  generateAiBtn.textContent = "✦ Generating...";

  try {
    // ── Gather profile fields for server-side ICP scoring ──
    const profilesAll = (await getLS("profiles")) || {};
    const currentProfile = profilesAll[urlKey] || {};

    const profileData = {
      headline:    currentProfile.headline    || document.getElementById("p_headline")?.innerText  || "",
      position:    currentProfile.position    || "",
      location:    currentProfile.location    || document.getElementById("p_company_loc")?.innerText || "",
      industry:    currentProfile.industry    || "",
      company:     currentProfile.company     || "",
      companySize: currentProfile.companySize || ""
    };

    // ── Persist raw text immediately (so it's not lost) ──
    if (urlKey) {
      const rec = profilesAll[urlKey] || {
        profileUrl: urlKey,
        dateCaptured: new Date().toISOString(),
        sentDates: { msg1: null, msg2: null, msg3: null }
      };
      rec.rawProfileText = rawText;
      rec.lastUpdated = new Date().toISOString();
      profilesAll[urlKey] = rec;
      await setLS({ profiles: profilesAll });
    }

    // ── Call backend ──
    const res = await fetch("https://api.linkedvue.com/ai/generate", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "Authorization": `Bearer ${session}`
      },
      body: JSON.stringify({
        rawProfileText: rawText,
        profileData,
        icpSettings
      })
    });

    const data = await res.json();

    if (!res.ok || !data.ok) {
      showToast("AI error: " + (data.error || "Unknown"));
      return;
    }

    // ── Update UI ──
    if (aiMessageBoxEl)   aiMessageBoxEl.value       = data.message  || "";
    if (icpScoreValueEl)  icpScoreValueEl.textContent = data.icpScore ?? "—";

    // ── Persist results to local profile record ──
    const profilesUpdated = (await getLS("profiles")) || {};
    const rec = profilesUpdated[urlKey] || {
      profileUrl: urlKey,
      dateCaptured: new Date().toISOString(),
      sentDates: { msg1: null, msg2: null, msg3: null }
    };

    rec.aiMessage      = data.message;
    rec.icpScore       = data.icpScore;
    rec.aiGeneratedAt  = new Date().toISOString();
    rec.lastUpdated    = new Date().toISOString();

    profilesUpdated[urlKey] = rec;
    await setLS({ profiles: profilesUpdated });

    showToast("✓ AI message ready");

  } catch (err) {
    console.error("[AI] Error:", err);
    showToast("AI request failed — check connection");
  } finally {
    generateAiBtn.disabled = false;
    generateAiBtn.textContent = "✦ Generate AI Message";
  }
});
});
