// sync-client.js — LinkedVue Sync Client
// Reads your existing chrome.storage.local format and syncs to backend

const SYNC_CONFIG = {
  apiBase: 'https://api.linkedvue.com',
  stalePullMs: 4 * 60 * 60 * 1000,  // pull if last pull > 4 hours ago
  sessionKey: 'linkedvue_session_id',
  lastSyncKey: 'linkedvue_last_sync',
  lastPullKey: 'linkedvue_last_pull',
};

// ─── SESSION ──────────────────────────────────────────────────────────────────

async function getSession() {
  const data = await chrome.storage.local.get(SYNC_CONFIG.sessionKey);
  return data[SYNC_CONFIG.sessionKey] || null;
}

async function isAuthenticated() {
  const session = await getSession();
  return !!session;
}

async function saveSession(sessionId) {
  await chrome.storage.local.set({
    [SYNC_CONFIG.sessionKey]: sessionId,
    [SYNC_CONFIG.lastPullKey]: 0,
  });
}

async function clearSession() {
  await chrome.storage.local.remove([
    SYNC_CONFIG.sessionKey,
    SYNC_CONFIG.lastSyncKey,
    SYNC_CONFIG.lastPullKey,
  ]);
}

// ─── API CALL ─────────────────────────────────────────────────────────────────

async function apiCall(method, path, body = null) {
  const session = await getSession();
  if (!session) throw new Error('Not authenticated');

  const options = {
    method,
    headers: {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${session}`,
    },
  };

  if (body) options.body = JSON.stringify(body);

  const response = await fetch(`${SYNC_CONFIG.apiBase}${path}`, options);
  const data = await response.json();

  if (!response.ok) throw new Error(data.error || 'API error');
  return data;
}

// ─── BUILD PUSH PAYLOAD ──────────────────────────────────────────────────────
// Reads your existing storage format and converts to API format

async function buildPushPayload(triggeredBy = 'scheduled') {
  const storage = await chrome.storage.local.get([
    'profiles',
    'templates',
    'icpSettings'
  ]);

  const rawProfiles = storage.profiles || {};
  const rawTemplates = storage.templates || {};
  const icpSettings = storage.icpSettings || null;

  // Convert profiles object to array for API
  const profiles = Object.entries(rawProfiles).map(([url, p]) => ({
    profile_url: url,
    first_name: p.firstName || null,
    last_name: p.lastName || null,
    email: p.email || null,
    notes: p.notes || null,
    status: p.status || null,
    icp_score: p.icpScore || null,
    raw_profile_text: p.rawProfileText || null,
    ai_message: p.aiMessage || null,
    template_id: p.templateUsed || null,
    sequence_delay_days: p.sequenceDelayDays || null,
    next_due_date: p.nextDueDate ? new Date(p.nextDueDate).getTime() : null,
    created_at: p.dateCaptured ? new Date(p.dateCaptured).getTime() : Date.now(),
    updated_at: p.lastUpdated ? new Date(p.lastUpdated).getTime() : Date.now(),
    // Convert sentDates to messages format for API
    messages: {
      msg1: { text: p.messages?.msg1 || null, sentAt: p.sentDates?.msg1 ? new Date(p.sentDates.msg1).getTime() : null },
      msg2: { text: p.messages?.msg2 || null, sentAt: p.sentDates?.msg2 ? new Date(p.sentDates.msg2).getTime() : null },
      msg3: { text: p.messages?.msg3 || null, sentAt: p.sentDates?.msg3 ? new Date(p.sentDates.msg3).getTime() : null },
    }
  }));

  // Convert templates object to array for API
  const templates = Object.entries(rawTemplates).map(([name, t]) => ({
    id: t.id || name,
    name: t.name || name,
    delay_days: t.delayDays || 3,
    created_at: t.createdAt ? new Date(t.createdAt).getTime() : Date.now(),
    updated_at: t.updatedAt ? new Date(t.updatedAt).getTime() : Date.now(),
    messages: {
      msg1: t.messages?.msg1 || null,
      msg2: t.messages?.msg2 || null,
      msg3: t.messages?.msg3 || null,
    }
  }));

  // Convert ICP settings
  const icp = icpSettings ? {
    industries: icpSettings.industries || [],
    regions: icpSettings.regions || [],
    personas: icpSettings.personas || [],
    goal: icpSettings.goal || null,
    offer: icpSettings.offer ? JSON.stringify(icpSettings.offer) : null,
  } : null;

  return {
    profiles,
    templates,
    icp_settings: icp,
    triggered_by: triggeredBy
  };
}

// ─── PUSH ─────────────────────────────────────────────────────────────────────

async function push(triggeredBy = 'scheduled') {
  try {
    const auth = await isAuthenticated();
    if (!auth) {
      console.log('[LinkedVue Sync] Not authenticated — skipping push');
      return { skipped: true };
    }

    const payload = await buildPushPayload(triggeredBy);

    if (payload.profiles.length === 0 &&
        payload.templates.length === 0 &&
        !payload.icp_settings) {
      console.log('[LinkedVue Sync] Nothing to push');
      return { skipped: true };
    }

    const result = await apiCall('POST', '/sync/push', payload);

    await chrome.storage.local.set({
      [SYNC_CONFIG.lastSyncKey]: Date.now()
    });

    console.log('[LinkedVue Sync] Push complete:', result.synced);
    return result;

  } catch (err) {
    console.error('[LinkedVue Sync] Push failed:', err.message);
    return { error: err.message };
  }
}

// ─── PULL ─────────────────────────────────────────────────────────────────────

async function pull() {
  try {
    const auth = await isAuthenticated();
    if (!auth) return { skipped: true };

    const lastPullData = await chrome.storage.local.get(SYNC_CONFIG.lastPullKey);
    const since = lastPullData[SYNC_CONFIG.lastPullKey] || 0;

    const result = await apiCall('GET', `/sync/pull?since=${since}`);

    if (result.profiles?.length > 0) {
      // Get existing profiles to merge into
      const existing = await chrome.storage.local.get('profiles');
      const currentProfiles = existing.profiles || {};

      for (const profile of result.profiles) {
        if (profile.deleted_at) {
          delete currentProfiles[profile.profile_url];
          continue;
        }

        // Convert API format back to your extension's format
        const existingRecord = currentProfiles[profile.profile_url] || {};

        currentProfiles[profile.profile_url] = {
          ...existingRecord,
          profileUrl: profile.profile_url,
          firstName: profile.first_name,
          lastName: profile.last_name,
          email: profile.email,
          notes: profile.notes,
          status: profile.status,
          icpScore: profile.icp_score,
          rawProfileText: profile.raw_profile_text,
          aiMessage: profile.ai_message,
          templateUsed: profile.template_id,
          sequenceDelayDays: profile.sequence_delay_days,
          nextDueDate: profile.next_due_date ? new Date(profile.next_due_date).toISOString() : null,
          dateCaptured: existingRecord.dateCaptured || new Date(profile.created_at).toISOString(),
          lastUpdated: new Date(profile.updated_at).toISOString(),
          // Restore messages
          messages: {
            msg1: profile.messages.find(m => m.slot === 1)?.content || null,
            msg2: profile.messages.find(m => m.slot === 2)?.content || null,
            msg3: profile.messages.find(m => m.slot === 3)?.content || null,
          },
          // Restore sentDates — never overwrite if already set locally
          sentDates: {
            msg1: existingRecord.sentDates?.msg1 || (profile.messages.find(m => m.slot === 1)?.sent_at ? new Date(profile.messages.find(m => m.slot === 1).sent_at).toISOString() : null),
            msg2: existingRecord.sentDates?.msg2 || (profile.messages.find(m => m.slot === 2)?.sent_at ? new Date(profile.messages.find(m => m.slot === 2).sent_at).toISOString() : null),
            msg3: existingRecord.sentDates?.msg3 || (profile.messages.find(m => m.slot === 3)?.sent_at ? new Date(profile.messages.find(m => m.slot === 3).sent_at).toISOString() : null),
          }
        };
      }

      await chrome.storage.local.set({ profiles: currentProfiles });
      console.log('[LinkedVue Sync] Pull merged:', result.profiles.length, 'profiles');
    }

    await chrome.storage.local.set({
      [SYNC_CONFIG.lastPullKey]: result.server_time
    });

    return result;

  } catch (err) {
    console.error('[LinkedVue Sync] Pull failed:', err.message);
    return { error: err.message };
  }
}

// ─── PULL IF STALE ────────────────────────────────────────────────────────────

async function pullIfStale() {
  const auth = await isAuthenticated();
  if (!auth) return;

  const data = await chrome.storage.local.get(SYNC_CONFIG.lastPullKey);
  const lastPull = data[SYNC_CONFIG.lastPullKey] || 0;
  const isStale = Date.now() - lastPull > SYNC_CONFIG.stalePullMs;

  if (isStale) {
    console.log('[LinkedVue Sync] Data is stale — pulling from server');
    await pull();
  }
}

// ─── EVENT PUSH (debounced) ───────────────────────────────────────────────────

let pushDebounceTimer = null;

function pushEvent(triggeredBy = 'event') {
  clearTimeout(pushDebounceTimer);
  pushDebounceTimer = setTimeout(() => {
    push(triggeredBy);
  }, 5000);
}

// ─── SCHEDULED SYNC ──────────────────────────────────────────────────────────

async function scheduledSync() {
  const auth = await isAuthenticated();
  if (!auth) return;

  console.log('[LinkedVue Sync] Running scheduled sync');
  await push('scheduled');
  pull();
}

// Export
const syncClient = {
  push,
  pull,
  pullIfStale,
  pushEvent,
  scheduledSync,
  isAuthenticated,
  saveSession,
  clearSession,
  getSession,
};